import 'package:admin_desktop/app_constants.dart';
import 'package:admin_desktop/application/bookings/booking_provider.dart';
import 'package:admin_desktop/application/bookings/booking_state.dart';
import 'package:admin_desktop/application/bookings/details/booking_details_provider.dart';
import 'package:admin_desktop/domain/models/data/booking_data.dart';
import 'package:admin_desktop/infrastructure/services/app_helpers.dart';
import 'package:admin_desktop/infrastructure/services/time_service.dart';
import 'package:admin_desktop/infrastructure/services/tr_keys.dart';
import 'package:admin_desktop/presentation/components/buttons/view_more_button.dart';
import 'package:admin_desktop/presentation/components/custom_checkbox.dart';
import 'package:admin_desktop/presentation/components/login_button.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/comments/widgets/mini_button.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:remixicon_updated/remixicon_updated.dart';

import '../components/list_top_bar.dart';

class BookingTable extends StatefulWidget {
  const BookingTable({super.key});

  @override
  State<BookingTable> createState() => _BookingTableState();
}

class _BookingTableState extends State<BookingTable> {
  @override
  Widget build(BuildContext context) {
    return LayoutBuilder(builder: (context, constraints) {
      return Consumer(builder: (context, ref, child) {
        final notifier = ref.read(bookingProvider.notifier);
        final state = ref.watch(bookingProvider);
        return SingleChildScrollView(
          child: Padding(
            padding: const EdgeInsets.symmetric(horizontal: 16),
            child: Column(
              children: [
                20.verticalSpace,
                _topBar(state: state, ref: ref),
                _topRow(constraints, ref),
                state.isLoading
                    ? _loading()
                    : state.bookings.isEmpty
                        ? Column(
                            children: [
                              80.verticalSpace,
                              Text(
                                AppHelpers.getTranslation(TrKeys.empty),
                                style: Style.interMedium(),
                              ),
                            ],
                          )
                        : Column(
                            children: [
                              ...state.bookings.map((e) {
                                return _tableRow(e, constraints, ref);
                              })
                            ],
                          ),
                if (state.hasMore)
                  ViewMoreButton(
                    onTap: () {
                      notifier.fetchBookings(
                        context: context,
                      );
                    },
                  ),
              ],
            ),
          ),
        );
      });
    });
  }

  Widget _topBar({required BookingState state, required WidgetRef ref}) {
    return Row(
      children: [
        ListTopBar(
            title: TrKeys.all,
            count: state.totalCount,
            onTap: () {
              ref.read(bookingProvider.notifier).changeTabIndex(0, context);
            },
            isLoading: state.isLoading,
            isActive: state.selectTabIndex == 0,
            onRefresh: () {
              ref
                  .read(bookingProvider.notifier)
                  .fetchBookings(context: context, isRefresh: true);
            }),
        ListTopBar(
            title: TrKeys.newKey,
            count: state.totalCount,
            onTap: () {
              ref
                  .read(bookingProvider.notifier)
                  .changeTabIndex(1, context, status: "new");
            },
            isLoading: state.isLoading,
            isActive: state.selectTabIndex == 1,
            onRefresh: () {
              ref.read(bookingProvider.notifier).fetchBookings(
                  context: context, isRefresh: true, status: "new");
            }),
        ListTopBar(
            title: TrKeys.booked,
            count: state.totalCount,
            onTap: () {
              ref
                  .read(bookingProvider.notifier)
                  .changeTabIndex(2, context, status: "booked");
            },
            isLoading: state.isLoading,
            isActive: state.selectTabIndex == 2,
            onRefresh: () {
              ref.read(bookingProvider.notifier).fetchBookings(
                  context: context, isRefresh: true, status: "booked");
            }),
        ListTopBar(
            title: TrKeys.progress,
            count: state.totalCount,
            onTap: () {
              ref
                  .read(bookingProvider.notifier)
                  .changeTabIndex(3, context, status: "progress");
            },
            isLoading: state.isLoading,
            isActive: state.selectTabIndex == 3,
            onRefresh: () {
              ref.read(bookingProvider.notifier).fetchBookings(
                  context: context, isRefresh: true, status: "progress");
            }),
        ListTopBar(
            title: TrKeys.ended,
            count: state.totalCount,
            onTap: () {
              ref
                  .read(bookingProvider.notifier)
                  .changeTabIndex(4, context, status: "ended");
            },
            isLoading: state.isLoading,
            isActive: state.selectTabIndex == 4,
            onRefresh: () {
              ref.read(bookingProvider.notifier).fetchBookings(
                  context: context, isRefresh: true, status: "ended");
            }),
        ListTopBar(
            title: TrKeys.canceled,
            count: state.totalCount,
            onTap: () {
              ref
                  .read(bookingProvider.notifier)
                  .changeTabIndex(5, context, status: "canceled");
            },
            isLoading: state.isLoading,
            isActive: state.selectTabIndex == 5,
            onRefresh: () {
              ref.read(bookingProvider.notifier).fetchBookings(
                  context: context, isRefresh: true, status: "canceled");
            }),
        Spacer(),
        ref.watch(bookingProvider).calendarType == 3
            ? LoginButton(
                bgColor: Style.black,
                icon: Icon(
                  Remix.add_circle_line,
                  color: Style.white,
                  size: 24.r,
                ),
                title: TrKeys.createBooking,
                onPressed: () =>
                    ref.read(bookingProvider.notifier).changeStateIndex(1),
              )
            : SizedBox.shrink(),
      ],
    );
  }

  Widget _tableRow(
    BookingData booking,
    BoxConstraints constraints,
    WidgetRef ref,
  ) {
    return GestureDetector(
      onTap: () {
        ref.read(bookingDetailsProvider.notifier).setBookingDetail(booking,
            success: () {
          ref.read(bookingProvider.notifier).changeStateIndex(5);
        });
      },
      child: Container(
        width: double.infinity,
        height: 60,
        decoration: BoxDecoration(
            color: Style.white,
            border: Border.fromBorderSide(BorderSide(color: Style.greyColor))),
        child: Row(
          mainAxisAlignment: MainAxisAlignment.start,
          children: [
            12.horizontalSpace,
            CustomCheckbox(
              isActive:
                  ref.watch(bookingProvider).selectBooking.contains(booking.id),
              onTap: () {
                ref.read(bookingProvider.notifier).selectBooking(
                    orderLength:
                        ref.watch(bookingProvider).selectBooking.length,
                    id: booking.id);
              },
            ),
            8.horizontalSpace,
            SizedBox(
              width: 30,
              child: Text("${booking.id}"),
            ),
            16.horizontalSpace,
            SizedBox(
                width: constraints.maxWidth / 5,
                child: Text(booking.user?.firstname ?? "  -")),
            16.horizontalSpace,
            SizedBox(
                width: constraints.maxWidth / 6,
                child: Text(
                    booking.serviceMaster?.service?.translation?.title ?? " ")),
            16.horizontalSpace,
            SizedBox(
              width: constraints.maxWidth / 8,
              child: Text(AppHelpers.numberFormat(number: booking.price)),
            ),
            16.horizontalSpace,
            SizedBox(
              width: constraints.maxWidth / 11,
              child: Row(
                children: [
                  Container(
                    alignment: Alignment.center,
                    margin: EdgeInsets.symmetric(vertical: 16),
                    padding: EdgeInsets.symmetric(horizontal: 12),
                    decoration: BoxDecoration(
                      color: AppHelpers.getStatusColor(booking.status)
                          .withOpacity(0.2),
                      borderRadius: BorderRadius.circular(AppConstants.radius),
                    ),
                    child: Text(
                      AppHelpers.getTranslation(booking.status ?? ""),
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                      style: Style.interRegular(
                          color: AppHelpers.getStatusColor(booking.status)),
                    ),
                  ),
                ],
              ),
            ),
            16.horizontalSpace,
            if (constraints.maxWidth > 1350)
              SizedBox(
                width: constraints.maxWidth / 6,
                child: Text(TimeService.dateFormatMDYHm(booking.createdAt)),
              ),
            16.horizontalSpace,
            Spacer(),
            MiniButton(
              icon: Remix.eye_line,
              onTap: () {
                ref
                    .read(bookingDetailsProvider.notifier)
                    .setBookingDetail(booking, success: () {
                  ref.read(bookingProvider.notifier).changeStateIndex(5);
                });
              },
            ),
            16.horizontalSpace,
          ],
        ),
      ),
    );
  }

  Container _topRow(BoxConstraints constraints, WidgetRef ref) {
    return Container(
      width: double.infinity,
      height: 60,
      decoration: BoxDecoration(
        color: Style.white,
        border: Border.fromBorderSide(
          BorderSide(color: Style.greyColor),
        ),
      ),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.start,
        children: [
          12.horizontalSpace,
          CustomCheckbox(
            isActive: ref.watch(bookingProvider).isAllSelect,
            onTap: () {
              ref
                  .read(bookingProvider.notifier)
                  .allSelectBooking(ref.watch(bookingProvider).bookings);
            },
          ),
          8.horizontalSpace,
          SizedBox(
              width: 30,
              child: Text(
                AppHelpers.getTranslation(TrKeys.id),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 5,
              child: Text(
                AppHelpers.getTranslation(TrKeys.user),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 6,
              child: Text(
                AppHelpers.getTranslation(TrKeys.service),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 8,
              child: Text(
                AppHelpers.getTranslation(TrKeys.price),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          SizedBox(
              width: constraints.maxWidth / 11,
              child: Text(
                AppHelpers.getTranslation(TrKeys.status),
                style: Style.interMedium(),
              )),
          16.horizontalSpace,
          if (constraints.maxWidth > 1350)
            SizedBox(
                width: constraints.maxWidth / 6,
                child: Text(
                  AppHelpers.getTranslation(TrKeys.date),
                  style: Style.interMedium(),
                )),
          16.horizontalSpace,
          Spacer(),
        ],
      ),
    );
  }

  ListView _loading() {
    return ListView.builder(
        itemCount: 10,
        shrinkWrap: true,
        itemBuilder: (context, index) {
          return Container(
            height: 60,
            width: double.infinity,
            decoration: BoxDecoration(
                color: Style.greyColor,
                border: Border.fromBorderSide(BorderSide(color: Style.white))),
          );
        });
  }
}
